/**
 * A function that can be called to remove an event listener.
 */
export type DisposeListener = () => void;
/**
 * An event handler that can be registered to listen for a specific event type.
 */
export type EventHandler<EventMap extends Record<string, object>, K extends keyof EventMap> = (eventArgs: EventMap[K]) => void | Promise<void>;
/**
 * Interface for adding and removing event listeners to an object that can raise certain events specified by {@link EventMap}.
 *
 * @typeParam EventMap An object type that represents the events that this API can emit (keys) and the shape of their event args (types).
 */
export interface IEventManager<EventMap extends Record<string, object>> {
    /**
     * Adds an event listener for the specified event type.
     *
     * @param eventType The type of event to listen for.
     * @param listener The event listener to add.
     * @returns A function that can be called to remove the event listener.
     */
    addEventListener<K extends keyof EventMap>(eventType: K, listener: EventHandler<EventMap, K>): DisposeListener;
    /**
     * Removes an event listener for the specified event type.
     *
     * @param eventType The type of event to remove the listener from.
     * @param listener The event listener to remove.
     */
    removeEventListener<K extends keyof EventMap>(eventType: K, listener: EventHandler<EventMap, K>): void;
}
/**
 * Helper class that can be inheritted from for event management. It also has a listener for all events that are emitted.
 */
export declare abstract class EventManager<EventArgsMap extends Record<string, object>> implements IEventManager<EventArgsMap> {
    private readonly listeners;
    onEmitEvent?: (eventType: string, eventArgs: object) => unknown;
    addEventListener<K extends keyof EventArgsMap>(eventType: K, listener: EventHandler<EventArgsMap, K>): DisposeListener;
    removeEventListener<K extends keyof EventArgsMap>(eventType: K, listener: EventHandler<EventArgsMap, K>): void;
    protected emitEvent<K extends keyof EventArgsMap>(eventType: K, eventArgs: EventArgsMap[K]): void;
}
//# sourceMappingURL=EventManager.d.ts.map