import { Big } from "big.js";

export interface ObjectValidation {
  /**
   * Returns the `GUID` of the validated {@link mendix/lib/MxObject}.
   */
  getGuid(): GUID;
  /**
   * Returns attributes which did not pass validation.
   */
  getAttributes(): Array<{ name: string; reason: string }>;
  /**
   * @see ObjectValidation#getAttributes
   */
  getFields(): Array<{ name: string; reason: string }>;
}

/**
 * A Mendix Object: {@link mendix/lib/MxObject MxObject}.
 *
 * @classdesc
 * An instance of an entity in the domain model of the current session.
 *
 * It provides methods for accessing and modifying attributes.
 *
 * Instances of `MxObject` should seldom be instantiated manually.
 * They are the result of calls into {@link mx.data} such as {@link mx.data.get} or
 * {@link mx.data.create}.
 */
export interface MxObject {
  /**
   * Returns the value of an attribute.
   *
   * @param {string} attr attribute whose value to return
   * @returns {string|number|external:Big|boolean} value of the specified attribute.
   * The return type depends on the type of the attribute as follows:
   *
   * | Attribute type         | Javascript type      | Javascript value                     |
   * |:-----------------------|:---------------------|:-------------------------------------|
   * | Integer, Long, Decimal | {@link external:Big} | instance containing the number       |
   * | Autonumber             | `string`             | string representation of the number  |
   * | DateTime               | `number`             | timestamp of the date                |
   * | Enum                   | `string`             | value of the enumeration             |
   * | String                 | `string`             |                                      |
   * | Boolean                | `boolean`            |                                      |
   * | Reference              | `string`             | referenced object `GUID` as a string |
   * | ReferenceSet           | `Array<string>`      | referenced object `GUIDs` as strings |
   *
   * Empty values are always returned as the empty string (`""`).
   */
  get<T extends ClientValue>(attribute: Attribute): T;
  /**
   * Returns the original, last committed, value of an attribute.
   *
   * @param {string} attr attribute whose value to return
   * @returns {string|number|external:Big|boolean} original value of the specified attribute.
   *
   * @see MxObject#get for an overview of how the return value depends on the
   * attribute's type.
   */
  getOriginalValue<T extends ClientValue>(attribute: Attribute): T;
  /**
   * Sets the value of attribute `attr` to the value of `val`.
   *
   * The type of `val` depends on the type of attribute.
   *
   * This only changes the {@link mendix/lib/MxObject} instance in the client session.  To
   * commit the object to the backend database, see {@link mx.data.commit}.
   *
   * @param {string} attr attribute to set
   * @param {*} val value to set
   * @throws Error if `value` is invalid, use validator#validate before calling this method
   */
  set(attribute: Attribute, value: ClientValue): void;
  /**
   * If `attr` is a reference attribute, sets it to the given object.
   * If `attr` is a reference set attribute, adds the given object to it.
   *
   * @param attr {string} The reference attribute.
   *                      Can refer to either a reference or reference set attribute.
   * @param guid {GUID} `GUID` of the object to add to the reference
   * @returns {boolean} `true` if successful, `false` otherwise
   */
  addReference(attribute: Attribute, value: ClientValue): boolean;
  /**
   * Checks whether an attribute is read-only.
   *
   * @param {string} attr attribute for which to check
   * @returns {boolean} `true` if `attr` is a read-only attribute, `false` otherwise
   */
  isReadonlyAttr(attribute: Attribute): boolean;
  /**
   * Convenience function
   *
   * @see MxMetaObject#isReference
   */
  isReference(attribute: Attribute): boolean;
  /**
   * Convenience function
   *
   * @see MxMetaObject#getAttributes
   */
  getAttributes(): Attribute[];
  /**
   * Retrieves the `MxObjects` referenced by a reference or reference set attribute.
   *
   * @param {string} attr reference attribute whose referenced objects to return
   */
  getReferences(attribute: Attribute): GUID[];
  /**
   * Retrieves the original `MxObjects` referenced by a reference or reference set attribute.
   *
   * @param {string} attr reference attribute whose original referenced objects to return
   */
  getOriginalReferences(attribute: Attribute): GUID[];
  /**
   * Returns the `GUID` of this `MxObject`.
   */
  getGuid(): GUID;
  /**
   * Gets the entity name.
   */
  getEntity(): Entity;
  /**
   * Convenience function
   *
   * @see MxMetaObject#getEnumCaption
   */
  getEnumCaption(attribute: Attribute, key: string): string | null;
  /**
   * Convenience function
   *
   * @see MxMetaObject#getEnumMap
   */
  getEnumMap(
    attribute: Attribute
  ): Array<{ key: string; caption: string }> | null;
  /**
   * Gets an object or value of an attribute, through a path from this object.
   *
   * The result is passed to a callback and depends on the requested path:
   *
   * | Path resolves to     | Result
   * |-------------------------------------------------------------------------------------
   * | attribute            | value of the attribute
   * | object reference     | {@link GUID} of the object
   * | object reference set | array of `GUIDs` with the objects
   * | entity               | array of `GUIDs` for a reference set, `MxObject` otherwise
   *
   * If no path is given, the result is set to the object itself.
   * A reference set can only be the last reference in the path.
   *
   * @param {string} path path from the object to the desired object or attribute
   * @param {(obj: any) => void} callback called when fetching is done
   * @param {(error: Error) => void} error function to handle errors
   */
  fetch(
    path: EntityPath | Attribute | AttributePath,
    callback: (result: AttributeValue | MxObject | MxObject[]) => void,
    error?: (e: Error) => void
  ): any;
}

/**
 * Retrieve and manipulate {@link mendix/lib/MxObject MxObjects}.
 */
export interface Data {
  /**
   * Retrieves {@link mendix/lib/MxObject MxObjects} from the Runtime (in online apps) or from the local database
   * (in offline apps). Note that not every parameter is supported in offline apps.
   *
   * Only one of the guid / guids / xpath / microflow parameters should be set.
   */
  get(
    args:
      | GetByGuidArgs
      | GetByMultipleGuidsArgs
      | GetByEntityArgs
      | GetByPathArgs
  ): MxObject;

  /**
   * Creates an {@link mendix/lib/MxObject MxObject}.
   *
   * @param {Object} args
   * @param {string} args.entity entity to create an instance of
   * @param {CreateSuccessCallback} args.callback
   *        function to handle the result when successful
   * @param {CreateErrorCallback} args.error function to handle errors
   * @param {Object} scope
   *        scope in which the error and callback handler are invoked
   */
  create(
    args: {
      entity: Entity;
      callback: (obj: MxObject) => void;
      error?: (e: Error) => void;
    },
    scope?: object
  ): void;

  /**
   * Commits an {@link mendix/lib/MxObject MxObject}.
   *
   * When there are changes, these will automatically be sent to the Runtime.
   *
   * @param {Object} args
   * @param {MxObject} args.mxobj `MxObject` to save changes for
   * @param {MxObject[]} args.mxobjs `MxObjects` to save changes for
   * @param {() => void} args.callback
   *        function to handle the result when successful
   * @param {(error: Error) => void} args.error function to handle errors
   * @param {(validations: ObjectValidation[]) => void} args.onValidation
   *        function to handle validation feedback
   * @param {Object} scope scope in which the error and callback handler are invoked
   */
  commit(
    args: {
      mxobjs: MxObject[];
      callback: () => void;
      error?: (error: Error) => void;
      onValidation?: (validations: ObjectValidation[]) => void;
    },
    scope?: object
  ): void;

  /**
   * Rollbacks an MxObject.
   *
   * @param {Object} args
   * @param {MxObject} args.mxobj `MxObject` to rollback changes for
   * @param {MxObject[]} args.mxobjs `MxObjects` to rollback changes for
   * @param {() => void} args.callback
   *        function to handle the result when successful
   * @param {(e: Error) => void} args.error function to handle errors
   * @param {Object} scope scope in which the error and callback handler are invoked
   */
  rollback(
    args: {
      mxobjs: MxObject[];
      callback: () => void;
      error?: (e: Error) => void;
    },
    scope?: object
  ): void;

  /**
   * Calls listeners to changes in a MxObject, an attribute of a MxObject or any changes to
   * MxObjects of a entity.
   *
   * @param {Object} args
   * @param {GUID} args.guid `GUID` to invoke the update for
   * @param {string} args.entity entity to invoke the update for
   * @param {string} args.attr attribute to invoke the update for
   * @param {() => void)} args.callback
   *        function to be called after all invocations have been done
   */
  update(args: {
    guid: GUID;
    entity: string;
    attr: string;
    callback: () => void;
  }): void;
}

export interface GetByGuidArgs {
  guid: GUID;
  error?: (e: Error) => void;
  callback: (obj: MxObject | null) => void;
}

export interface GetByMultipleGuidsArgs {
  guids: GUID[];
  error?: (e: Error) => void;
  callback: (objs: MxObject[]) => void;
}

export interface GetByEntityArgs {
  entity: Entity;
  filter: Partial<{
    offset: number;
    sort: string[][][];
    amount: number;
  }>;
  error?: (e: Error) => void;
  callback: (objs: MxObject[]) => void;
}

export interface GetByPathArgs {
  entity: Entity;
  path: EntityPath;
  guid: GUID;
  direction?: "direct" | "reverse";
  error?: (e: Error) => void;
  callback: (objs: MxObject[]) => void;
}

declare global {
  /**
   * Container of the Mendix client subsystems.
   */
  const mx: {
    /**
     * Root URL from which application is loaded, e.g. `https://domain.com/mendixapp/`.
     */
    appUrl: string;
    /**
     * URL of the server, e.g. `https://domain.com/mendixapp/`.
     */
    remoteUrl: string;
    /**
     * Retrieve and manipulate {@link mendix/lib/MxObject MxObjects}.
     */
    data: Data;
  };
}

type Attribute = string & { __attributeTag: any };
type AttributePath = string & { __attributePathTag: any };
type AttributeValue = undefined | string | boolean | Date | Big | GUID | GUID[];
type EntityPath = string & { __entityTag: any };
type GUID = string & { __guidTag: any };

type DateNumber = number; // cannot use tag, as it breaks `typeof === "number"` type guard

type ClientValue = string | boolean | DateNumber | Big | GUID | GUID[];

type Entity = string & { __entityTag: any };
