import { createWriteStream } from "fs";
import { EOL } from "os";
import { setTimeout } from 'timers';
import * as nodePath from "path";
import { watch, rollup } from "rollup";

const isProduction = process.env.NODE_ENV === "production" || false;
const logPath = process.env.MX_WEB_CLIENT_BUILD_LOG || "log.txt";
var logger = createWriteStream(logPath, { flags: "a" /* append */ });

let deploymentWebDirectory;
if (process.env.MX_DEPLOYMENT_WEB_DIRECTORY) {
    // Can be used for debugging purposes.
    // 1. Deploy the app once.
    // 2. Set MX_DEPLOYMENT_WEB_DIRECTORY to the app's deployment/web directory path
    // Now you can launch and debug this script from your Studio Pro installer
    deploymentWebDirectory = process.env.MX_DEPLOYMENT_WEB_DIRECTORY;
    process.chdir(deploymentWebDirectory);
} else {
    deploymentWebDirectory = process.cwd();
}

const nlogLevels = {
    info: "INFO",
    debug: "DEBUG",
    warn: "WARN",
    error: "ERROR",
    fatal: "FATAL",
};

const spCodes = {
    start: "START",
    success: "SUCCESS",
    error: "ERROR",
};

const spErrorTypes = {
    general: "GENERIC",
    jsAction: "JS_ACTION",
    widget: "WIDGET",
};

function log(level, message) {
    const timestamp = new Date().toISOString();
    // We strip ANSI control characters because they make log files less readable
    const cleanMessage = message.replace(/\u001b[^m]*?m/g, "");
    logger.write(timestamp + " " + level + " " + cleanMessage + EOL);
}

function formatError(message, error) {
    return message + EOL + error;
}

function out(code, payload) {
    process.stdout.write(JSON.stringify({ code, payload }) + EOL);
}

let config;
try {
    const { default: defaultImport } = await import(`file://` + deploymentWebDirectory + "/rollup.config.mjs");
    config = defaultImport;
} catch (err) {
    out(spCodes.error, err.toString());
    log(nlogLevels.fatal, formatError("Failed to load configuration file", err));
    process.abort();
}

if (isProduction) {
    try {
        log(nlogLevels.info, "Build started");
        out(spCodes.start);
        log(nlogLevels.info, "Bundling started");
        const bundle = await rollup({
            ...config,
            onwarn: (warning) => {
                log(nlogLevels.warn, warning.message);
            },
        });
        await bundle.write(config.output);
        out(spCodes.success);
        log(nlogLevels.info, "Bundling finished");
        log(nlogLevels.info, "Build stopped");
    } catch (err) {
        const formattedError = formatError("Build failed", err);
        
        const file = err?.loc?.file;
        const fileName = file ? getFileName(file) : null;

        out(spCodes.error, {
            message: formattedError,
            type: getErrorType(err),
            fileName,
        });
        log(nlogLevels.error, formattedError);
    }

    await new Promise((resolve) => setTimeout(resolve, 1000)); // Delay exit a bit so SP can process the final status
} else {
    const watcher = watch({
        ...config,
        onwarn: (warning) => {
            log(nlogLevels.warn, warning.message);
        },
    });

    watcher.on("event", (e) => {
        switch (e.code) {
            case "START": {
                log(nlogLevels.info, "Build started");
                break;
            }
            case "BUNDLE_START": {
                out(spCodes.start);
                log(nlogLevels.info, "Bundling started");
                break;
            }
            case "BUNDLE_END": {
                out(spCodes.success);
                log(nlogLevels.info, "Bundling finished in " + e.duration + " milliseconds");
                break;
            }
            case "ERROR": {
                const errorMessage = formatRollupError(e.error);
                const errorType = getErrorType(e.error);

                const file = e.error?.loc?.file;
                const fileName = file ? getFileName(file) : null;

                out(spCodes.error, {
                    type: errorType,
                    message: errorMessage,
                    fileName,
                });
                log(nlogLevels.error, formatError("Bundling failed", errorMessage));
                break;
            }
            case "END": {
                log(nlogLevels.info, "Build stopped");
                break;
            }
        }

        if (e.result) {
            e.result.close();
        }
    });
}

function getErrorType(error) {
    const filePath = error?.loc?.file;
    if (!filePath) {
        return spErrorTypes.general;
    }

    if (filePath.includes("javascriptsource") && filePath.includes("actions")) {
        return spErrorTypes.jsAction;
    }

    if (filePath.includes(nodePath.join("web", "widgets"))) {
        return spErrorTypes.widget;
    }

    return spErrorTypes.general;
}

function getFileName(path) {
    return path.split(nodePath.sep).pop();
}

function formatRollupError(error) {
    const name = error.name || error.cause?.name;
    const nameSection = name ? `${name}: ` : "";
    const pluginSection = error.plugin ? `(plugin ${error.plugin}) ` : "";
    const message = `${pluginSection}${nameSection}${error.message}`;

    const outputLines = [message.toString()];

    if (error.url) {
        outputLines.push(error.url);
    }

    if (error.loc) {
        outputLines.push(`${error.loc.file || error.id} (${error.loc.line}:${error.loc.column})`);
    } else if (error.id) {
        outputLines.push(error.id);
    }

    if (error.frame) {
        outputLines.push(error.frame);
    }

    if (error.stack) {
        outputLines.push(error.stack?.replace(`${nameSection}${error.message}\n`, ""));
    }

    // ES2022: Error.prototype.cause is optional
    if (error.cause) {
        let cause = error.cause;
        const causeErrorLines = [];
        let indent = "";

        while (cause) {
            indent += "  ";
            const message = cause.stack || cause;
            causeErrorLines.push(...`[cause] ${message}`.split("\n").map((line) => indent + line));

            cause = cause.cause;
        }

        outputLines.push(causeErrorLines.join("\n"));
    }

    outputLines.push("", "");

    return outputLines.join("\n");
}
