import crypto from "crypto";
import fs from "fs";
import path from "path";

/**
 * Finds all files that need to be pre-cached and generates service-worker.js
 */
export default function serviceWorker(options) {
    return {
        name: "rollup-plugin-mendix-serviceworker",
        async writeBundle(_, bundles) {
            const serviceWorkerPath = path.join(options.deploymentDir, "web", "service-worker.js");

            const precacheEntries = {
                html: await findStaticDeploymentFiles(options.deploymentDir, ".html"),
                generic: (
                    await findStaticDeploymentFiles(
                        options.deploymentDir,
                        ".json",
                        ".css",
                        ".js",
                        ".woff2",
                        ".woff",
                        ".svg"
                    )
                ),
                profiles: [],
            };

            const additionalPrecacheEntries = await Promise.all(
                Object.keys(bundles).map((p) => generateCacheEntry(options.deploymentDir, path.join("dist", p)))
            );

            let text = (await fs.promises.readFile(serviceWorkerPath)).toString();
            text = text.replace("INJECTED_PRECACHE_ENTRIES", JSON.stringify(precacheEntries));
            text = text.replace("ADDITIONAL_PRE_CACHE_ENTRIES", JSON.stringify(additionalPrecacheEntries));
            await fs.promises.writeFile(serviceWorkerPath, text);
        },
    };
}

async function findStaticDeploymentFiles(deploymentDir, ...extensions) {
    const ignoredPaths = [
        "index.js",
        "layouts",
        "nanoflows",
        "package.json",
        "pages",
        "preview",
        "rollup.config.mjs",
        "service-worker.js",
        "widgets",
    ];
    const foundFiles = [];

    async function walkDir(dir) {
        const items = await fs.promises.readdir(path.join(deploymentDir, "web", dir), { withFileTypes: true });

        for (const item of items) {
            const localPath = path.join(dir, item.name);
            if (ignoredPaths.includes(localPath)) {
                continue;
            }

            if (item.isDirectory()) {
                await walkDir(localPath);
            } else if (extensions.some((e) => item.name.endsWith(e))) {
                foundFiles.push(localPath);
            }
        }
    }

    await walkDir("");

    return Promise.all(foundFiles.map((f) => generateCacheEntry(deploymentDir, f)));
}

async function generateCacheEntry(deploymentDir, file) {
    let revision = null;
    if (file.endsWith(".html") || file.endsWith(".woff2") || file.endsWith(".woff")) {
        const fullPath = path.join(deploymentDir, "web", file);
        const text = (await fs.promises.readFile(fullPath)).toString();
        revision = crypto.createHash("md5").update(text).digest("hex");
    }

    return {
        url: file.split(path.sep).join(path.posix.sep),
        revision,
    };
}
