import { dirname, extname, join, normalize, posix, relative, resolve, sep } from "path";
import { existsSync, readFileSync } from "fs";
import { globSync } from "glob";

class VariableDynamicImportError extends Error {
}

/* eslint-disable-next-line no-template-curly-in-string */
const example = "For example: import(`./foo/${bar}.js`).";

function sanitizeString(str) {
    if (str.includes("*")) {
        throw new VariableDynamicImportError("A dynamic import cannot contain * characters.");
    }
    return str;
}

function templateLiteralToGlob(node) {
    let glob = "";

    for (let i = 0; i < node.quasis.length; i += 1) {
        glob += sanitizeString(node.quasis[i].value.raw);
        if (node.expressions[i]) {
            glob += expressionToGlob(node.expressions[i]);
        }
    }

    return glob;
}

function callExpressionToGlob(node) {
    const { callee } = node;
    if (
        callee.type === "MemberExpression" &&
        callee.property.type === "Identifier" &&
        callee.property.name === "concat"
    ) {
        return `${expressionToGlob(callee.object)}${node.arguments.map(expressionToGlob).join("")}`;
    }
    return "*";
}

function binaryExpressionToGlob(node) {
    if (node.operator !== "+") {
        throw new VariableDynamicImportError(`${node.operator} operator is not supported.`);
    }

    return `${expressionToGlob(node.left)}${expressionToGlob(node.right)}`;
}

function expressionToGlob(node) {
    switch (node.type) {
        case "TemplateLiteral":
            return templateLiteralToGlob(node);
        case "CallExpression":
            return callExpressionToGlob(node);
        case "BinaryExpression":
            return binaryExpressionToGlob(node);
        case "Literal": {
            return sanitizeString(node.value);
        }
        default:
            return "*";
    }
}

function dynamicImportToGlob(node, sourceString) {
    let glob = expressionToGlob(node);
    if (!glob.includes("*")) {
        return null;
    }
    glob = glob.replace(/\*\*/g, "*");

    if (glob.startsWith("/")) {
        throw new VariableDynamicImportError(
            `invalid import "${sourceString}". Variable absolute imports are not supported, imports must start with ./ in the static part of the import. ${example}`
        );
    }

    if (glob.startsWith("./*.")) {
        throw new VariableDynamicImportError(
            `${`invalid import "${sourceString}". Variable imports cannot import their own directory, ` +
            "place imports in a separate directory or make the import filename more specific. "
            }${example}`
        );
    }

    if (extname(glob) === "") {
        throw new VariableDynamicImportError(
            `invalid import "${sourceString}". A file extension must be included in the static part of the import. ${example}`
        );
    }

    return glob;
}

const relativePath = /^\.?\.\//;
const absolutePath = /^(?:\/|(?:[A-Za-z]:)?[\\|/])/;

function isRelative(path) {
    return relativePath.test(path);
}

function isAbsolute(path) {
    return absolutePath.test(path);
}

const PAGES_FOLDER = "./pages";
const PAGES_GLOB = `${PAGES_FOLDER}/*.js?*`;

export default function mendixResolve(resolutionCachePath, contextPath) {
    let cache = {};

    function updateCache() {
        if (existsSync(resolutionCachePath)) {
            const resolutionCache = readFileSync(resolutionCachePath, "utf-8");
            const parsed = JSON.parse(resolutionCache);
            for (let key in parsed) {
                const newPart = {
                    ...parsed[key],
                    id: join(contextPath, parsed[key].id)
                };

                cache[join(contextPath, key)] = newPart;
                cache[key] = newPart;
                cache[key.replace(".js", "")] = newPart;

                if (key.endsWith("/index.js")) {
                    cache[key.replace("/index.js", "")] = newPart;
                }
            }
        }
    }

    updateCache();

    const ignoreResolve = [
        "\u0000",
        "?commonjs-require",
        "?commonjs-proxy",
        "\x00"
    ];

    return {
        name: "rollup-plugin-mendix-resolve",

        // Resolves direct dependencies inside the deployment folder, as well as node_module dependencies
        resolveId(source, importer) {
            if (ignoreResolve.some(r => source.includes(r))) {
                return null;
            }

            let key = source;

            // Try to resolve from cache
            key = importer && isRelative(key) ? resolve(dirname(importer), key) : key;
            const normalizedKey = normalize(key);

            if (typeof cache[key] !== "undefined") {
                return cache[key];
            } else if (typeof cache[normalizedKey] !== "undefined") {
                return cache[normalizedKey];
            }

            // Don't resolve package imports (e.g. react/mendix)
            // We also don't care about the normalizedKey here
            if (isAbsolute(key) || isRelative(key)) {
                const deploymentFolder = process.cwd();
                const relativePath = relative(deploymentFolder, key);

                // Only directly resolve sources that are in the app deployment folder
                if (relativePath.startsWith("..") || isAbsolute(relativePath)) {
                    return null;
                }

                return {
                    id: key.endsWith(".js") || key.endsWith(".mjs") || key.endsWith(".css") ? key : key + ".js"
                };
            }

            return null;
        },

        // Takes care of resolving the dynamic import for pages
        resolveDynamicImport(specifier) {
            if (typeof specifier == "string") {
                const importedFileName = specifier.split("/").reverse()[0];
                return join(contextPath, "mendix", importedFileName);
            }

            const deploymentFolder = process.cwd();
            const importGlob = dynamicImportToGlob(specifier);

            if(importGlob.startsWith("*")){
                // External dependency that needs to be loaded dynamically
                return false;
            }

            if (importGlob === PAGES_GLOB) {
                const pageSearchGlob = PAGES_GLOB.substring(0, PAGES_GLOB.indexOf("?"));
                const files = globSync(pageSearchGlob, { cwd: deploymentFolder });
                files.forEach((f) => {
                    const relativeFileLocation = relative(deploymentFolder, f).replace("\\", "/");

                    this.emitFile({
                        id: relativeFileLocation,
                        fileName: relativeFileLocation,
                        type: "chunk"
                    });

                    cache[relativeFileLocation] = {
                        "external": false,
                        "id": relativeFileLocation,
                        "meta": {},
                        "moduleSideEffects": true,
                        "syntheticNamedExports": false
                    };
                });
                return false;
            }
        },

        async buildStart() {
            this.addWatchFile(resolutionCachePath); // Add the resolution cache to the watch files when watching for development
            this.addWatchFile(PAGES_FOLDER); // Watches the pages folder in case new pages are exported
        },

        // Update cache when resolution cache is changed
        watchChange(id) {
            // the paths are registered as unix, but the id is in the platform specific path standard
            const unixPath = id.split(sep).join(posix.sep);

            switch (unixPath) {
                case resolutionCachePath:
                    updateCache();
                    break;
            }
        }
    };
};
