"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Storage = void 0;
var _index = require("./index");
/**
 * Creates a new async-storage api compatible instance.
 * The encryption key is only used when compiled against the SQLCipher version of op-sqlite.
 */
class Storage {
  constructor(options) {
    this.db = (0, _index.open)({
      ...options,
      name: '__opsqlite_storage'
    });
    this.db.executeSync('PRAGMA mmap_size=268435456');
    this.db.executeSync('CREATE TABLE IF NOT EXISTS storage (key TEXT PRIMARY KEY, value TEXT) WITHOUT ROWID');
  }
  async getItem(key) {
    const result = await this.db.execute('SELECT value FROM storage WHERE key = ?', [key]);
    let value = result.rows[0]?.value;
    if (typeof value !== 'undefined' && typeof value !== 'string') {
      throw new Error('Value must be a string or undefined');
    }
    return value;
  }
  getItemSync(key) {
    const result = this.db.executeSync('SELECT value FROM storage WHERE key = ?', [key]);
    let value = result.rows[0]?.value;
    if (typeof value !== 'undefined' && typeof value !== 'string') {
      throw new Error('Value must be a string or undefined');
    }
    return value;
  }
  async setItem(key, value) {
    await this.db.execute('INSERT OR REPLACE INTO storage (key, value) VALUES (?, ?)', [key, value.toString()]);
  }
  setItemSync(key, value) {
    this.db.executeSync('INSERT OR REPLACE INTO storage (key, value) VALUES (?, ?)', [key, value.toString()]);
  }
  async removeItem(key) {
    await this.db.execute('DELETE FROM storage WHERE key = ?', [key]);
  }
  removeItemSync(key) {
    this.db.executeSync('DELETE FROM storage WHERE key = ?', [key]);
  }
  async clear() {
    await this.db.execute('DELETE FROM storage');
  }
  clearSync() {
    this.db.executeSync('DELETE FROM storage');
  }
  getAllKeys() {
    return this.db.executeSync('SELECT key FROM storage').rows.map(row => row.key);
  }
}
exports.Storage = Storage;
//# sourceMappingURL=Storage.js.map